/*
 * FableTools.cpp
 *
 * This file implements the classes needed to manipulate Fable Wad files.
 *
 * First things first, mad props to Fatum who put theroy into practice.  The guts of these 
 * classes are derived from his code.  I'm just making it look pretty and easy to use.  Also,
 * having a class means you don't have to worry about changing your code when we find out 
 * something new.  Just download the new class and compile it!
 * -bigfreak
 * 
 * Big Thanks to Amins and Mods at http://fable.lanpirates.net for bringing us together.
 * 
 * irc.dynastynet.net
 * #fablehax
 */

#include <fstream.h>
#include <string.h>
#include <direct.h>


#include "genList.h"
#include "fableWad.h"

/*
 *	GetMaxFileSize takes the size of a file in bytes and the blocksize and returns the 
 *	maxsize that file can be.
 */
DWORD GetMaxFileSize(DWORD dwSize, DWORD dwBlockSize)
{
	DWORD dwRet = 0;
	DWORD dwRounded = dwSize / dwBlockSize;
	if((dwBlockSize * dwRounded) == dwSize)
	{
		//it just so happens that dwSize is a multiple of dwBlockSize
		//this is the rare case where the file will fit exactly in a number of blocks
		return dwBlockSize * dwRounded;
	}

	return ((dwSize / dwBlockSize) + 1) * dwBlockSize;
}


/*
 * CFileInfo::CFileInfo and CFileInfo::~CFileInfo 
 *   basic construtors and destructors.
 */
CFileInfo::CFileInfo()
{
	fileName[0] = 0;
	dwSize = 0;
	dwAddress = 0;
	dwHeader = 0;
	dwMaxsize = 0;
	inputFilename[0] = 0;
}
CFileInfo::CFileInfo(const char* filename, DWORD inSize, 
				DWORD inAddress, DWORD inHeader, DWORD inMaxsize)
{
	strcpy(fileName, filename);
	dwSize = inSize;
	dwAddress = inAddress;
	dwHeader = inHeader;
	dwMaxsize = inMaxsize;
	inputFilename[0] = 0;
}
CFileInfo::~CFileInfo()
{
	//not a lot to do here yet
}

/*
 * CFileInfo::Copy makes a copy of itself and returns a pointer to it
 *   the caller must delete[] it
 */
CFileInfo* CFileInfo::Copy()
{
	CFileInfo* pRet = new CFileInfo();
	strcpy(pRet->fileName, fileName);
	pRet->dwSize = dwSize;
	pRet->dwAddress = dwAddress;
	pRet->dwHeader = dwHeader;
	pRet->dwMaxsize = dwMaxsize;
	strcpy(pRet->inputFilename, inputFilename);
	return pRet;
}

/*
 * CFileInfo::QueueFile this function is used to flag this file as being
 *		queued for importing into the new wad.
 *
 */
void CFileInfo::QueueFile(const char* inputFile)
{
	if(inputFile == NULL) return;
	strcpy(inputFilename, inputFile);
}

/*
 * CWad::CWad and CWad::~CWad basic construction and destruction
 *
 */
CWad::CWad()
{
	fileName[0] = 0;
	bOpen = FALSE;
}
CWad::~CWad()
{
	//not much to do here as GenList's destructor will clean all allocated memory
}

/*
 * CWad::Open tries to open the wad file pointed to by inputWadFile.
 *   It processes the file list and returns TRUE on success and FALSE on failure.
 */
BOOL CWad::Open(const char* inputWadFile)
{
	if(inputWadFile == NULL) return FALSE;
	//copy the filename for the prime function
	strcpy(fileName, inputWadFile);
	//try to prime the filelist
	if(!PrimeFileList())
	{
		//failed to init wad file so we're gonna reset back to 
		//  unopened state (clean up)
		Close();
		return FALSE;
	}
	//all systems go!
	bOpen = TRUE;
	return TRUE;
}

/*
 * CWad::Close resets all data stored for the wad file and reverts the class back
 *		to an unopened state.
 */
void CWad::Close()
{
	fileList.RemoveAll();	//cleans up all the memory allocated by the file list
	fileName[0] = 0;
	bOpen = FALSE;	
}

/*
 * CWad::PrimeFileList is a private member function which opens the wad and tries to build
 *	a list of files in contains
 */
BOOL CWad::PrimeFileList()
{
	ifstream infile;
	infile.open(fileName, ios::binary);
	if (!infile)
		return FALSE;
	infile.seekg(0);

	WADHEADER wadHeader;

	//read in the wadheader
	infile.read((char*) &wadHeader, sizeof(wadHeader));

	//move to wadHeader.start
	infile.seekg(wadHeader.start);

	// vars to store file info
	char fileName[2048];
	DWORD dwSize;
	DWORD dwAddress;
	DWORD dwHeader;
	DWORD dwMaxsize;

	//file header holders
	FILEHEADER_A fileHeaderA;
	FILEHEADER_B fileHeaderB;

	// loop through the files enumerating their info
	for (int i = 0; i < wadHeader.ilosc; i++)
	{
		//store our current address in the file
		dwHeader = infile.tellg();

		//read in fileheader part A
		infile.read((char*) &fileHeaderA, sizeof(fileHeaderA));
		//grab more data about the file
		dwAddress = fileHeaderA.adress;
	    dwSize = fileHeaderA.size;
		//the following code *should* work for maxsize, but we'll see
		// --bigfreak
		dwMaxsize = GetMaxFileSize(dwSize, wadHeader.blocksize);
		//read the filename
		infile.read(fileName, fileHeaderA.namesize);
		//null terminate the string
		fileName[fileHeaderA.namesize] = 0;
		//read in fileheader part B
		infile.read((char*) &fileHeaderB, sizeof(fileHeaderB));

		//make a file info object
		CFileInfo* pNewFile = new CFileInfo(
										fileName, 
										dwSize, 
										dwAddress, 
										dwHeader, 
										dwMaxsize);
		//stick it at the end of the list
		if(pNewFile != NULL)
			fileList.AddTail((void*)pNewFile);
	}
	return TRUE;
}

/*
 * CWad::FileCount returns the number of files in the wad
 */
DWORD CWad::FileCount()
{
	return fileList.GetCount();
}

/*
 * CWad::GetFile returns a file references by dwIndex (0 -> FileCount-1)
 *     Set bCopy to TRUE if you'd like an object returned that you can delete yourself
 */
CFileInfo* CWad::GetFile(DWORD dwIndex, BOOL bCopy/* = FALSE*/)
{
	CFileInfo* pRet = (CFileInfo*) fileList[dwIndex];
	if(bCopy)
	{
		pRet = pRet->Copy();
	}
	return pRet;
}

/*
 * CWad::ExtractFile extracts the given file from the wad.  The caller must reference
 *		the index number of the file and the output filename.
 */
BOOL CWad::ExtractFile(DWORD dwIndex, const char* outFilename)
{
	//get fileinfo object;
	CFileInfo* pFileInfo = (CFileInfo*)fileList[dwIndex];
	if((pFileInfo == NULL) || (outFilename == NULL)) return FALSE;

	//open the wad for input
	ifstream infile;
	infile.open(fileName, ios::binary);
	if (!infile)
		return FALSE;

	//open the output file
	ofstream outfile;
	outfile.open(outFilename, ios::binary);
	if (!outfile)
		return FALSE;
	char buforek[2048];
	int wielkosc = 0;
	wielkosc = pFileInfo->dwSize;
	infile.seekg(pFileInfo->dwAddress);
	do
	{
		infile.read((char*) &buforek, sizeof(buforek));
		if (wielkosc >= sizeof(buforek))
		{
			outfile.write((char*) &buforek, sizeof(buforek));
			wielkosc = wielkosc - sizeof(buforek);
		}
		else
		{
			outfile.write((char*) &buforek, wielkosc);
			wielkosc = 0;
		}
	}
	while (wielkosc > 0);
	infile.close();
	outfile.close();
	return TRUE;
}

/*
 * CWad::InjectFile injects a given file into the wad.  The caller must reference
 *		the index number of the file to overwrite and the input filename.
 */
BOOL CWad::InjectFile(DWORD dwIndex, const char* inFilename)
{
	//get fileinfo object;
	CFileInfo* pFileInfo = (CFileInfo*)fileList[dwIndex];
	if((pFileInfo == NULL) || (inFilename == NULL)) return FALSE;
	
	//see if the input file is too big
    ifstream impfile(inFilename, ios::binary);
    if (!impfile)
      return FALSE;
    impfile.seekg(0, ios::end);
    DWORD impsize = impfile.tellg();
	DWORD maxsize = pFileInfo->dwMaxsize;
	if(impsize > maxsize)
	{
		impfile.close();
		return FALSE;
	}

	//open the wad for injection
	fstream iofile(fileName, ios::binary | ios::in | ios::out);
	if (!iofile)
		return FALSE;

	//move back to the begining of the input file
    impfile.seekg(0);

	//change the filesize in the wad 
    char buforek[2048];
    iofile.seekg(pFileInfo->dwHeader + 0x18);
    iofile.write((char*) &impsize, sizeof(impsize));
    //change the filesize in our list of files
	pFileInfo->dwSize = impsize;

	//goto the pos where we need to inject the new file
    iofile.seekg(pFileInfo->dwAddress);
    DWORD impwielkosc = impsize;
    while (!impfile.eof())
    {
		impfile.read((char*) &buforek, sizeof(buforek));
		if (impwielkosc < sizeof(buforek))
			//read the remainder of the file
			iofile.write((char*) &buforek, impwielkosc);
		else
			iofile.write((char*) &buforek, sizeof(buforek));
		impwielkosc = impwielkosc - sizeof(buforek);
    }
	
    //can we use memset?
	memset(buforek, 0x00, sizeof(buforek));
//	for (int j = 0; j < sizeof(buforek); j++)
//    buforek[j] = 0x00;

	//we need to fill the remainder of the allocated space in the wad with 0's
    impwielkosc = maxsize - impsize;
    do
    {
		if (impwielkosc >= sizeof(buforek))
		{
			iofile.write((char*) &buforek, sizeof(buforek));
			impwielkosc = impwielkosc - sizeof(buforek);
		}
		else
		{
			iofile.write((char*) &buforek, impwielkosc);
			impwielkosc = 0;
		}
    }
    while (impwielkosc > 0);
    iofile.close();
    impfile.close();
	return TRUE;
}

/*
 * CWad::InjectLargeFile will immediately rebuild the wad so that it contains the new, larger
 *		file.  Caller must supply, the index number of the file to be replaced, the input
 *		filename and the output filename for the resulting wad. 
 */
BOOL CWad::InjectLargeFile(DWORD dwIndex, const char* inFilename, const char* outputWadFile)
{
	if(!QueueInjection(dwIndex, inFilename))
		return FALSE;
	return DumpWad(outputWadFile);
}

/*
 * CWad::QueueInjection queues a file to be injected into the wad.  The caller doesn't have to
 *		know if the file is too big.  The caller must provide the index number of the replaced 
 *		file and the filename of the input file
 */
BOOL CWad::QueueInjection(DWORD dwIndex, const char* inFilename)
{
	CFileInfo* pFileInfo = (CFileInfo*)fileList[dwIndex];
	if((pFileInfo == NULL) || (inFilename == NULL)) return FALSE;
	pFileInfo->QueueFile(inFilename);
	return TRUE;
}

/*
 * CWad::DumpWad rebuilds the entire wad file and injects any queued files along the way.  The
 *		caller must supply an output filename for the resulting wad.
 */	
BOOL CWad::DumpWad(const char* outputWadFile)
{
	//the following code is experimental
	/*
		In order to increase a filesize past maxsize we must add a given number of blocks to 
		end of the space allocated for the original file.

		As we add those blocks, we change several bits of data:

		1.) the start of each file after that is shifted by the number of bytes we added
			therefore each fileHeaderA.adress we must offset to point to the new data
		2.) further more, the list of file headers lives at the end of the wad past
			all the files and past the data we've inserted.  Therefore, the offset which
			points to the block of file	headers starts must also be alterd to reflect 
			(wadHeader.start)

		To keep track of the number of blocks added or removed, we will have curBlockOffset  This
		number will be positive to show we've added a block or negative to show we've removed a block
		as we progress down the file adding and removing blocks curBlockOffset will change to 
		reflect our current offset state.  For example:

		File A needs 5 more blocks -> curBlockOffset = 5
		File B needs 6 less blocks -> curBlockOffset -= 6;  or curBlockOffset = -1

		When after we go through all the files, it will be time to write out the file headers.  
		We will change the header information we have stored as we go through the files.  To write
		the new header information we will simply recall that saved data and update the file 
		headers with the new up to date data.

		The position of this file header block is up in the air until we have gone through the 
		entire list of files.  Therefore, we'll have to come back later and update that starting 
		address in the file.
	 */

	//let's get started 8-)

	//open our input and output files
	fstream iofile(outputWadFile, ios::binary | ios::in | ios::out);
	ifstream infile(fileName, ios::binary);

    if (!iofile || !infile)
      return FALSE;

	//quickly write the old header to the new file
	//	we'll come back later to fix up the offset for the file headers
	char buffer[0x800];
	WADHEADER wadHeader;
	//well there is exactly 1 block allocated to the header (0x800 bytes)
	infile.read(buffer, 0x800);
	iofile.write(buffer, 0x800);
	//but our wadheader struct is much smaller so we going to copy the header data
	//  in from the 0x800 byte block
	memcpy(&wadHeader, buffer, sizeof(wadHeader));

	//this is a new file list we'll use to track file data in the newly created wad
	GenList newFileList;

	//loop through the files seeing if we need to import a new file
	DWORD dwCount = fileList.GetCount();
	//this keeps track our the num blocks added or removed
	long curBlockOffset = 0;
	for(DWORD i = 0; i < dwCount; i++)
	{
		CFileInfo* pCurFile = (CFileInfo*)fileList[i];
		
		// make a copy of this data and stick it in our new file list
		CFileInfo* pNewFile = pCurFile->Copy();
		newFileList.AddTail(pNewFile);

		//adjust the offset information which will be written out later
		pNewFile->dwAddress += curBlockOffset * (long)wadHeader.blocksize;
		pNewFile->dwHeader = 0xFFFFFFFF;	//header location will change but I don't need it

		if(pCurFile->inputFilename[0] != 0)
		{
			//we have a file to input here

			//get the filesize of the new file
			ifstream inputFile;
			inputFile.open(pCurFile->inputFilename, ios::binary);
			inputFile.seekg(0, ios::end);
			DWORD dwSize = inputFile.tellg();
			inputFile.seekg(0, ios::beg);

			DWORD dwMaxSize = GetMaxFileSize(dwSize, wadHeader.blocksize);
			DWORD dwBlocks = dwMaxSize / wadHeader.blocksize;
			DWORD dwOldBlocks = pCurFile->dwMaxsize / wadHeader.blocksize;
			
			//loop through loading the input file
			for(DWORD j = 0; j < dwBlocks; j++)
			{
				//write zero's to the buffer initially
				memset(buffer, 0x00, wadHeader.blocksize);
				//read in a block of data from the input file
				inputFile.read(buffer, wadHeader.blocksize);
				//note: if we've read past the end of the file the rest of the buffer will 
				//		contain the zero's we previously set with memset
				//write out the buffer to the new wad
				iofile.write(buffer, wadHeader.blocksize);
			}
			//close up the input file
			inputFile.close();

			//we now need to advance the input wad file down past the overwritten file
			infile.seekg(pCurFile->dwMaxsize, ios::cur);

			//now we need to store the new data 8-D
			pNewFile->dwMaxsize = dwMaxSize;
			pNewFile->dwSize = dwSize;

			//need to keep track of blocks added or removed (fixes up the offsets)
			long blockDiff = (long)dwBlocks - (long)dwOldBlocks;
			curBlockOffset += blockDiff;
		}
		else
		{
			//this file is untouched we only need to read it and write it
			DWORD numWrites = pCurFile->dwMaxsize / wadHeader.blocksize;
			for(DWORD j = 0; j < numWrites; j++)
			{
				infile.read(buffer, wadHeader.blocksize);
				iofile.write(buffer, wadHeader.blocksize);
			}
		}
		//as we dump each of these files to the new wad, we're going to remove them from the queue
		pCurFile->inputFilename[0] = 0;
	}
	
	//Now that we've written out all the file data to the new wad, now we need to write out the 
	//  header info.  We also need to update the wadHeader with the address

	//get our location so we can update the wadHeader
	WADHEADER newWadHeader;
	memcpy(&newWadHeader, &wadHeader, sizeof(newWadHeader));
	newWadHeader.start = iofile.tellg();
	//update the file
	iofile.seekg(0, ios::beg);
	iofile.write((char*)&newWadHeader, sizeof(newWadHeader));
	//move back to where we left off
	iofile.seekg(newWadHeader.start, ios::beg);

	//now we need to update the file headers
	for(i = 0; i < dwCount; i++)
	{
		//read in the current file header
		FILEHEADER_A fileHeaderA;
		infile.read((char*)&fileHeaderA, sizeof(fileHeaderA));
		//need to make our alterations (sizes and addresses)
		CFileInfo* pCurFile = (CFileInfo*)newFileList[i];
		fileHeaderA.size = pCurFile->dwSize;
		fileHeaderA.adress = pCurFile->dwAddress;
		//write that puppy out to the new wad
		iofile.write((char*)&fileHeaderA, sizeof(fileHeaderA));
		//read in the filename
		char filename[2048];
		infile.read(filename, fileHeaderA.namesize);
		//write out the filename
		iofile.write(filename, fileHeaderA.namesize);
		//read in the rest of the file header
		FILEHEADER_B fileHeaderB;
		infile.read((char*)&fileHeaderB, sizeof(fileHeaderB));
		// and then write it out to the new Wad
		iofile.write((char*)&fileHeaderB, sizeof(fileHeaderB));
	}

	//read in and spit out those last 12 unknown bytes *shrug*
	infile.read(buffer, 12);
	iofile.write(buffer, 12);


	//close input and output files
	iofile.close();
	infile.close();

	return TRUE;
}

/*
 * CWad::DirectoryExport dumps the entire contents of the wad to a directory.  Any sub directories
 *		that do not exist will be created for you.  The caller must give the full path to the
 *		output directory and it must not include a trailing backslash.
 */
BOOL CWad::DirectoryExport(const char* dirPath)
{
	DWORD dwCount = fileList.GetCount();
	char tmpFileName[2048];
	char tmpDir[2048];
	char output[2048];
	for(DWORD i = 0; i<dwCount; i++)
	{
		CFileInfo* pCurFile = (CFileInfo*)fileList[i];
		
		//make all the needed direcoties and build filename
		strcpy(tmpFileName, pCurFile->fileName);
		char* pCur = tmpFileName;
		char* pPos = strstr(pCur, "\\");
		strcpy(output, dirPath);
		while(pPos != NULL)
		{
			memcpy(tmpDir, pCur, pPos-pCur);
			tmpDir[pPos-pCur] = 0;
			strcat(output, "\\");
			strcat(output, tmpDir);
			_mkdir(output);
			pCur = pPos + 1;
			pPos = strstr(pCur, "\\");
		}
		strcat(output, "\\");
		strcat(output, pCur);

		//export this file
		if(!ExtractFile(i, output))
			return FALSE;
	}
	return TRUE;
}


BOOL CWad::DirectoryImport(const char* dirPath)
{
	//will implement as needed
	return TRUE;
}

/*
 *  CWad::GetIndex gives back the index number of the file in the referenced to by file
 */
DWORD CWad::GetIndex(const char* file)
{
	DWORD dwCount = fileList.GetCount();
	for(DWORD i = 0; i < dwCount; i++)
	{
		CFileInfo* pCurFile = (CFileInfo*)fileList[i];
		char* pChar = strstr(pCurFile->fileName, file);
		if(pChar != NULL)
		{
			if(pCurFile->fileName == pChar)
			{
				return i;
			}
			char* oneChar = ((char*)pChar-1);
			if( oneChar[0] == '\\' )
			{
				return i;
			}
		}
	}
	return 0xFFFFFFFF;
}


/*
 *  CWad::InjectFile is the same as it's sister function but takes a filename instead of an index
 */
BOOL CWad::InjectFile(const char* replaceFile, const char* inFilename)
{
	DWORD dwIndex = GetIndex(replaceFile);
	if(dwIndex == 0xFFFFFFFF)
		return FALSE;
	return InjectFile(dwIndex, inFilename);
}

/*
 *  CWad::InjectLargeFile is the same as it's sister function but takes a filename instead of an index
 */
BOOL CWad::InjectLargeFile(
						   const char* replaceFile, 
						   const char* inFilename, 
						   const char* outputWadFile)
{
	DWORD dwIndex = GetIndex(replaceFile);
	if(dwIndex == 0xFFFFFFFF)
		return FALSE;
	return InjectLargeFile(dwIndex, inFilename, outputWadFile);
}

/*
 *  CWad::QueueInjection is the same as it's sister function but takes a filename instead of an index
 */
BOOL CWad::QueueInjection(const char* replaceFile, const char* inFilename)
{
	DWORD dwIndex = GetIndex(replaceFile);
	if(dwIndex == 0xFFFFFFFF)
		return FALSE;
	return QueueInjection(dwIndex, inFilename);
}

/*
 *  CWad::ExtractFile is the same as it's sister function but takes a filename instead of an index
 */
BOOL CWad::ExtractFile(const char* extractFile, const char* outFilename)
{
	DWORD dwIndex = GetIndex(extractFile);
	if(dwIndex == 0xFFFFFFFF)
		return FALSE;
	return ExtractFile(dwIndex, outFilename);
}

